<?php

declare(strict_types=1);

namespace MSML\MediaManager\ValueObjects;

use MSML\MediaManager\Rules\Rule;
use MSML\MediaManager\Rules\RuleContext;
use MSML\MediaManager\Rules\RulesEngine;
use MSML\MediaManager\Rules\EngineResult;
use MSML\MediaManager\Rules\MaxFilesRule;
use MSML\MediaManager\Models\MediaNamespace;
use MSML\MediaManager\Rules\MaxFileSizeRule;
use MSML\MediaManager\Rules\AllowRootFilesRule;
use MSML\MediaManager\Rules\ImageDimensionsRule;
use MSML\MediaManager\Rules\AllowedMimeTypesRule;
use MSML\MediaManager\Rules\AllowedExtensionsRule;

final class NamespaceConfiguration
{
    /** @var Rule[] */
    private array $rules = [];

    private string|null $disk;

    private RulesEngine|null $engine = null;

    /**
     * @var array<string, array{
     *     width: int,
     *     height: int,
     *     fit: 'contain'|'crop'|'fill'|'max'|'stretch',
     *     quality: int,
     *     format: ?string
     * }> $conversions
     */
    private array $conversions = [];

    private bool $hidden = false;

    private function __construct(private string $name) {}

    public static function create(string $name): self
    {
        $instance = new self($name);

        /** @var MediaNamespace|null $namespace */
        $namespace = MediaNamespace::whereSlug($name)->first();
        if ($namespace) {
            $instance->disk = $namespace->disk;
        }

        return $instance;
    }

    public function disk(string $disk): self
    {
        $this->disk = $disk;

        return $this;
    }

    public function addRule(Rule $rule): self
    {
        $this->rules[] = $rule;

        return $this;
    }

    public function maxFileSize(int $bytes): self
    {
        return $this->addRule(new MaxFileSizeRule($bytes));
    }

    /**
     * @param  string[]  $mimeTypes
     */
    public function allowedMimeTypes(array $mimeTypes): self
    {
        return $this->addRule(new AllowedMimeTypesRule($mimeTypes));
    }

    /**
     * @param  string[]  $extensions
     */
    public function allowedExtensions(array $extensions): self
    {
        return $this->addRule(new AllowedExtensionsRule($extensions));
    }

    public function disallowRootFiles(): self
    {
        return $this->addRule(new AllowRootFilesRule(false));
    }

    public function maxFiles(int $max): self
    {
        return $this->addRule(new MaxFilesRule($max));
    }

    public function maxImageDimensions(int $width, int $height): self
    {
        return $this->addRule(new ImageDimensionsRule($width, $height));
    }

    public function validateAction(RuleContext $context): EngineResult
    {
        if (empty($this->rules)) {
            return EngineResult::success();
        }

        return $this->getEngine()->evaluate($this->rules, $context);
    }

    private function getEngine(): RulesEngine
    {
        if ($this->engine === null) {
            $this->engine = new RulesEngine;
        }

        return $this->engine;
    }

    public function hidden(bool $hidden = true): self
    {
        $this->hidden = $hidden;

        return $this;
    }

    public function isHidden(): bool
    {
        return $this->hidden;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function getDisk(): string
    {
        if (!isset($this->disk)) {
            throw new \RuntimeException('Disk is not set for namespace: ' . $this->name);
        }

        return $this->disk;
    }

    /**
     * @param array<string, array{
     *     width: int,
     *     height: int,
     *     fit: 'contain'|'crop'|'fill'|'max'|'stretch',
     *     quality: int,
     *     format: ?string
     * }> $conversions
     */
    public function conversions(array $conversions): self
    {
        $this->conversions = array_merge($this->conversions, $conversions);

        return $this;
    }

    /**
     * @param  'contain'|'crop'|'fill'|'max'|'stretch'  $fit
     */
    public function addConversion(
        string $name,
        int $width,
        int $height,
        string $fit = 'crop',
        int $quality = 80,
        string|null $format = null
    ): self {
        $this->conversions[$name] = [
            'width'   => $width,
            'height'  => $height,
            'fit'     => $fit,
            'quality' => $quality,
            'format'  => $format,
        ];

        return $this;
    }

    /**
     * @return array<string, array{
     *    width: int,
     *    height: int,
     *    fit: 'contain'|'crop'|'fill'|'max'|'stretch',
     *    quality: int,
     *    format: ?string
     *    }>
     */
    public function getConversions(): array
    {
        return $this->conversions;
    }
}
