<?php

declare(strict_types=1);

namespace MSML\MediaManager\Actions;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use MSML\MediaManager\Models\Asset;
use Illuminate\Database\Query\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\JoinClause;
use MSML\MediaManager\Models\AssetFolder;
use MSML\MediaManager\Models\MediaNamespace;
use Illuminate\Pagination\LengthAwarePaginator;
use MSML\MediaManager\DataObjects\MediaBreadcrumb;
use MSML\MediaManager\DataObjects\MediaNavigation;
use MSML\MediaManager\DataObjects\MediaBrowserItem;
use MSML\MediaManager\Enums\MediaBrowserItemTypeEnum;
use MSML\MediaManager\DataObjects\MediaNavigationResponse;
use Illuminate\Database\Eloquent\Builder as EloquentBuilder;

final class HandleMediaNavigationAction
{
    public function handle(MediaNavigation $mediaNavigation): MediaNavigationResponse
    {
        if ($mediaNavigation->search) {
            return $this->handleGlobalSearch($mediaNavigation);
        }

        if ($mediaNavigation->isRootLevel()) {
            return $this->handleRootLevel($mediaNavigation);
        }

        if ($mediaNavigation->isNamespaceLevel()) {
            return $this->handleNamespaceLevel($mediaNavigation);
        }

        return $this->handleFolderLevel($mediaNavigation);
    }

    private function handleRootLevel(MediaNavigation $mediaNavigation): MediaNavigationResponse
    {
        $query = MediaNamespace::query()->withCount(['assets', 'folders']);

        $sortColumn = in_array($mediaNavigation->sortColumn, ['name', 'created_at'], true)
            ? $mediaNavigation->sortColumn
            : 'created_at';
        $direction = $mediaNavigation->sortDirection === 'asc' ? 'asc' : 'desc';

        $namespaces = $query->orderBy($sortColumn, $direction)
            ->paginate($mediaNavigation->perPage, page: $mediaNavigation->page);

        /** @var Collection<int, MediaBrowserItem> $items */
        $items = $namespaces->getCollection()->map(fn (MediaNamespace $namespace) => $this->mapNamespaceToItem($namespace));

        return new MediaNavigationResponse(
            currentPath: [],
            items: $items,
            pagination: $this->formatPagination($namespaces),
            canGoUp: false
        );
    }

    private function handleNamespaceLevel(MediaNavigation $mediaNavigation): MediaNavigationResponse
    {
        $namespace = MediaNamespace::whereSlug($mediaNavigation->namespaceSlug)->firstOrFail();

        $folderQ = AssetFolder::selectRaw("
            id,
            name,
            'folder' AS item_type,
            created_at,
            NULL AS file_name,
            NULL AS media_size,
            NULL AS mime_type
        ")
            ->where('media_namespace_id', $namespace->id)
            ->whereNull('parent_id');

        $assetQ = Asset::selectRaw("
            assets.id,
            COALESCE(m.file_name, 'Untitled') AS name,
            'asset' AS item_type,
            assets.created_at,
            m.file_name,
            m.size AS media_size,
            m.mime_type
        ")
            ->leftJoin('media as m', function (JoinClause $join): void {
                $join->on('assets.id', '=', 'm.model_id')
                    ->where('m.model_type', Asset::class);
            })
            ->where('media_namespace_id', $namespace->id)
            ->whereNull('asset_folder_id');

        [$items, $pagination] = $this->paginateAndMapItems($folderQ, $assetQ, $mediaNavigation);

        return new MediaNavigationResponse(
            currentPath: [
                new MediaBreadcrumb(name: $namespace->name, path: $namespace->slug),
            ],
            items: $items,
            pagination: $pagination,
            canGoUp: true
        );
    }

    private function handleFolderLevel(MediaNavigation $mediaNavigation): MediaNavigationResponse
    {
        $namespace = MediaNamespace::whereSlug($mediaNavigation->namespaceSlug)->firstOrFail();
        $folder = $this->resolveFolderFromPath($namespace, $mediaNavigation->folderPath);

        $folderQ = AssetFolder::selectRaw("
            id,
            name,
            'folder' AS item_type,
            created_at,
            NULL AS file_name,
            NULL AS media_size,
            NULL AS mime_type
        ")
            ->where('parent_id', $folder->id);

        $assetQ = Asset::selectRaw("
            assets.id,
            COALESCE(m.file_name, 'Untitled') AS name,
            'asset' AS item_type,
            assets.created_at,
            m.file_name,
            m.size AS media_size,
            m.mime_type
        ")
            ->leftJoin('media as m', function (JoinClause $join): void {
                $join->on('assets.id', '=', 'm.model_id')
                    ->where('m.model_type', Asset::class);
            })
            ->where('asset_folder_id', $folder->id);

        [$items, $pagination] = $this->paginateAndMapItems($folderQ, $assetQ, $mediaNavigation);

        $breadcrumbs = collect([
            new MediaBreadcrumb(name: $namespace->name, path: $namespace->slug),
        ]);

        $folder->getAncestors()->each(function (AssetFolder $ancestor) use ($breadcrumbs, $namespace): void {
            $breadcrumbs->push(new MediaBreadcrumb(
                name: $ancestor->name,
                path: $namespace->slug . '/' . implode('/', $ancestor->getPathSegments())
            ));
        });

        $breadcrumbs->push(new MediaBreadcrumb(
            name: $folder->name,
            path: $namespace->slug . '/' . implode('/', $folder->getPathSegments())
        ));

        return new MediaNavigationResponse(
            currentPath: $breadcrumbs->all(),
            items: $items,
            pagination: $pagination,
            canGoUp: true
        );
    }

    private function resolveFolderFromPath(MediaNamespace $namespace, string|null $folderPath): AssetFolder
    {
        if (!$folderPath) {
            throw new \InvalidArgumentException('Folder path cannot be empty');
        }

        $pathSegments = explode('/', trim($folderPath, '/'));
        $pathSegments = array_filter($pathSegments);

        $currentFolder = null;

        foreach ($pathSegments as $segment) {
            $currentFolder = AssetFolder::where('media_namespace_id', $namespace->id)
                ->where('parent_id', $currentFolder?->id)
                ->where('name', $segment)
                ->firstOrFail();
        }

        return $currentFolder ?? throw new \InvalidArgumentException('Could not resolve folder path');
    }

    /**
     * @param  LengthAwarePaginator<int, MediaNamespace>  $paginator
     * @return array{current_page: int, last_page: int, per_page: int, total: int}
     */
    private function formatPagination(LengthAwarePaginator $paginator): array
    {
        return [
            'current_page' => $paginator->currentPage(),
            'last_page'    => $paginator->lastPage(),
            'per_page'     => $paginator->perPage(),
            'total'        => $paginator->total(),
        ];
    }

    /**
     * @param  EloquentBuilder<AssetFolder>  $folderQuery
     * @param  EloquentBuilder<Asset>  $assetQuery
     * @return array{0: Collection<int, MediaBrowserItem>, 1: array{current_page: int, last_page: int, per_page: int, total: int}}
     */
    private function paginateAndMapItems(EloquentBuilder $folderQuery, EloquentBuilder $assetQuery, MediaNavigation $mediaNavigation): array
    {
        $union = $folderQuery->unionAll($assetQuery);
        $unionQuery = DB::table(DB::raw("({$union->toSql()}) AS u"))
            ->mergeBindings($union->getQuery());

        $sortedQuery = $this->applySortingToQuery($unionQuery, $mediaNavigation->sortColumn, $mediaNavigation->sortDirection);

        $paginated = $sortedQuery->paginate($mediaNavigation->perPage, ['*'], 'page', $mediaNavigation->page);

        /** @var Collection<int, object{id: int, name: string, item_type: string, created_at: \Illuminate\Support\Carbon, file_name?: string, media_size?: int, mime_type?: string}> $results */
        $results = $paginated->items();

        /** @var array<int> $folderIds */
        $folderIds = collect($results)->where('item_type', 'folder')->pluck('id')->all();

        /** @var array<int> $assetIds */
        $assetIds = collect($results)->where('item_type', 'asset')->pluck('id')->all();

        /** @var Collection<int, AssetFolder> $folders */
        $folders = AssetFolder::whereIn('id', $folderIds)->get()->keyBy('id');

        /** @var Collection<int, Asset> $assets */
        $assets = Asset::with('media')->whereIn('id', $assetIds)->get()->keyBy('id');

        /** @var Collection<int, MediaBrowserItem> $items */
        $items = collect($results)->map(function (object $result) use ($folders, $assets): MediaBrowserItem {
            return match ($result->item_type) {
                'folder' => $this->mapFolderToItem($folders->get($result->id)),
                'asset'  => $this->mapAssetToItem($assets->get($result->id)),
                default  => throw new \InvalidArgumentException('Unknown item type: ' . $result->item_type),
            };
        });

        $pagination = [
            'current_page' => $paginated->currentPage(),
            'last_page'    => $paginated->lastPage(),
            'per_page'     => $paginated->perPage(),
            'total'        => $paginated->total(),
        ];

        return [$items, $pagination];
    }

    private function handleGlobalSearch(MediaNavigation $mediaNavigation): MediaNavigationResponse
    {
        $search = $mediaNavigation->search;

        $namespacesQ = MediaNamespace::selectRaw("
            id,
            name,
            'namespace' AS item_type,
            created_at,
            CAST(NULL AS text) AS file_name,
            CAST(NULL AS bigint) AS media_size,
            CAST(NULL as text) AS mime_type,
            slug AS namespace_slug,
            CAST(NULL AS text) AS folder_path
        ")
            ->where(function ($query) use ($search) {
                $this->applyCaseInsensitiveSearch($query, 'name', $search);
                $query->orWhere(function ($subQuery) use ($search) {
                    $this->applyCaseInsensitiveSearch($subQuery, 'slug', $search);
                });
            });

        $foldersQ = AssetFolder::selectRaw("
            af.id,
            af.name,
            'folder' AS item_type,
            af.created_at,
            CAST(NULL AS text) AS file_name,
            CAST(NULL AS bigint) AS media_size,
            CAST(NULL as text) AS mime_type,
            mn.slug AS namespace_slug,
            af.name AS folder_path
        ")
            ->from('asset_folders as af')
            ->join('media_namespaces as mn', 'af.media_namespace_id', '=', 'mn.id')
            ->where(function ($query) use ($search) {
                $this->applyCaseInsensitiveSearch($query, 'af.name', $search);
            });

        $assetsQ = Asset::selectRaw("
            assets.id,
            COALESCE(m.file_name, 'Untitled') AS name,
            'asset' AS item_type,
            assets.created_at,
            m.file_name,
            m.size AS media_size,
            m.mime_type,
            mn.slug AS namespace_slug,
            af.name AS folder_path
        ")
            ->leftJoin('media as m', function (JoinClause $join): void {
                $join->on('assets.id', '=', 'm.model_id')
                    ->where('m.model_type', Asset::class);
            })
            ->join('media_namespaces as mn', 'assets.media_namespace_id', '=', 'mn.id')
            ->leftJoin('asset_folders as af', 'assets.asset_folder_id', '=', 'af.id')
            ->where(function ($query) use ($search) {
                $this->applyCaseInsensitiveSearch($query, 'm.file_name', $search);
            });

        $union = $namespacesQ->unionAll($foldersQ)->unionAll($assetsQ);

        $unionQuery = DB::table(DB::raw("({$union->toSql()}) AS search_results"))
            ->mergeBindings($union->getQuery());

        $sortedQuery = $this->applySortingToQuery($unionQuery, $mediaNavigation->sortColumn, $mediaNavigation->sortDirection);

        $paginated = $sortedQuery->paginate($mediaNavigation->perPage, ['*'], 'page', $mediaNavigation->page);

        /** @var Collection<int, object{id: int, name: string, item_type: string, created_at: \Illuminate\Support\Carbon, file_name?: string, media_size?: int, mime_type?: string}> $results */
        $results = $paginated->items();

        /** @var array<int> $namespaceIds */
        $namespaceIds = collect($results)->where('item_type', 'namespace')->pluck('id')->all();
        /** @var array<int> $folderIds */
        $folderIds = collect($results)->where('item_type', 'folder')->pluck('id')->all();
        /** @var array<int> $assetIds */
        $assetIds = collect($results)->where('item_type', 'asset')->pluck('id')->all();

        /** @var Collection<int, MediaNamespace> $namespaces */
        $namespaces = MediaNamespace::whereIn('id', $namespaceIds)->get()->keyBy('id');
        /** @var Collection<int, AssetFolder> $folders */
        $folders = AssetFolder::whereIn('id', $folderIds)->get()->keyBy('id');
        /** @var Collection<int, Asset> $assets */
        $assets = Asset::with('media')->whereIn('id', $assetIds)->get()->keyBy('id');

        /** @var Collection<int, MediaBrowserItem> $items */
        $items = collect($results)->map(function (object $result) use ($namespaces, $folders, $assets): MediaBrowserItem {
            return match ($result->item_type) {
                'namespace' => $this->mapNamespaceToItem($namespaces->get($result->id)),
                'folder'    => $this->mapFolderToItem($folders->get($result->id)),
                'asset'     => $this->mapAssetToItem($assets->get($result->id)),
                default     => throw new \InvalidArgumentException('Unknown item type: ' . $result->item_type),
            };
        });

        return new MediaNavigationResponse(
            currentPath: $mediaNavigation->isRootLevel() ? [] : $this->getCurrentPathFromNavigation($mediaNavigation),
            items: $items,
            pagination: [
                'current_page' => $paginated->currentPage(),
                'last_page'    => $paginated->lastPage(),
                'per_page'     => $paginated->perPage(),
                'total'        => $paginated->total(),
            ],
            canGoUp: false
        );
    }

    /**
     * @return array<MediaBreadcrumb>
     */
    private function getCurrentPathFromNavigation(MediaNavigation $mediaNavigation): array
    {
        if ($mediaNavigation->isRootLevel()) {
            return [];
        }

        $breadcrumbs = [];

        if ($mediaNavigation->namespaceSlug) {
            $namespace = MediaNamespace::whereSlug($mediaNavigation->namespaceSlug)->first();
            if ($namespace) {
                $breadcrumbs[] = new MediaBreadcrumb(name: $namespace->name, path: $namespace->slug);
            }
        }

        if ($mediaNavigation->folderPath) {
            $pathSegments = explode('/', trim($mediaNavigation->folderPath, '/'));
            $currentPath = $mediaNavigation->namespaceSlug;

            foreach ($pathSegments as $segment) {
                $currentPath .= '/' . $segment;
                $breadcrumbs[] = new MediaBreadcrumb(name: $segment, path: $currentPath);
            }
        }

        return $breadcrumbs;
    }

    private function mapNamespaceToItem(MediaNamespace|null $namespace): MediaBrowserItem
    {
        assert($namespace !== null);

        return new MediaBrowserItem(
            id: $namespace->id,
            name: $namespace->name,
            type: MediaBrowserItemTypeEnum::Namespace,
            path: $namespace->slug,
            size: null,
            thumbUrl: null,
            createdAt: $namespace->created_at,
            mimeType: null,
            metadata: []
        );
    }

    private function mapFolderToItem(AssetFolder|null $folder): MediaBrowserItem
    {
        assert($folder !== null);

        return new MediaBrowserItem(
            id: $folder->id,
            name: $folder->name,
            type: MediaBrowserItemTypeEnum::Folder,
            path: $folder->getPathSegments(),
            size: null,
            thumbUrl: null,
            createdAt: $folder->created_at,
            mimeType: null,
            metadata: []
        );
    }

    private function mapAssetToItem(Asset|null $asset): MediaBrowserItem
    {
        assert($asset !== null);

        return new MediaBrowserItem(
            id: $asset->id,
            name: $asset->original_name ?? 'Untitled',
            type: MediaBrowserItemTypeEnum::Asset,
            path: null,
            size: $asset->size,
            // TODO: handle conversions correctly
            thumbUrl: $asset->url,
            createdAt: $asset->created_at,
            mimeType: $asset->mime_type,
            metadata: [
                'mime_type' => $asset->mime_type,
                'file_name' => $asset->original_name,
            ]
        );
    }

    private function applySortingToQuery(Builder $query, string|null $sortColumn, string|null $sortDirection): Builder
    {
        if (!$sortColumn) {
            return $query->orderBy('created_at', 'desc');
        }

        $direction = $sortDirection === 'asc' ? 'asc' : 'desc';

        return match ($sortColumn) {
            'size' => $query->orderByRaw("
                CASE item_type
                    WHEN 'folder' THEN 1
                    WHEN 'namespace' THEN 2
                    WHEN 'asset' THEN 3
                END,
                media_size {$direction} NULLS LAST,
                created_at DESC
            "),

            'type' => $query->orderByRaw("
                CASE item_type
                    WHEN 'folder' THEN '0_folder'
                    WHEN 'namespace' THEN '0_namespace'
                    WHEN 'asset' THEN CONCAT('1_', COALESCE(mime_type, 'unknown'))
                END {$direction},
                created_at DESC
            "),

            'name' => $query->orderBy('name', $direction)
                ->orderBy('created_at', 'desc'),

            'created_at' => $query->orderBy('created_at', $direction),

            default => $query->orderBy('created_at', 'desc'),
        };
    }

    /**
     * Apply case-insensitive search to the query.
     *
     * @template T of Model
     *
     * @param  EloquentBuilder<T>  $query
     */
    private function applyCaseInsensitiveSearch(EloquentBuilder $query, string $column, string|null $search): void
    {
        $driver = config('database.default');
        assert(is_string($driver));
        $connection = config("database.connections.{$driver}.driver");

        if (!$search) {
            return;
        }

        if ($connection === 'pgsql') {
            $query->where($column, 'ILIKE', "%{$search}%");
        } else {
            $query->whereRaw("UPPER({$column}) LIKE UPPER(?)", ["%{$search}%"]);
        }
    }
}
