<?php

declare(strict_types=1);

namespace MSML\MediaManager\Rules;

use MSML\MediaManager\Enums\AssetAction;

final class AllowedExtensionsRule implements Rule
{
    /**
     * @var string[]
     */
    private readonly array $allowedExtensions;

    /**
     * @param  string[]  $allowedExtensions
     */
    public function __construct(array $allowedExtensions)
    {
        $this->allowedExtensions = array_map('strtolower', $allowedExtensions);
    }

    public function evaluate(RuleContext $context): RuleResult
    {
        return match ($context->action) {
            AssetAction::UPLOAD => $this->evaluateUpload($context),
            default             => RuleResult::skip(__('media-manager::rules.general.not_applicable'))
        };
    }

    private function evaluateUpload(RuleContext $context): RuleResult
    {
        if (empty($this->allowedExtensions)) {
            return RuleResult::skip(
                __('media-manager::rules.allowed_extensions.skip_no_restrictions')
            );
        }

        if ($context->filename === null) {
            return RuleResult::skip(
                __('media-manager::rules.allowed_extensions.skip_no_filename')
            );
        }

        $extension = strtolower(pathinfo($context->filename, PATHINFO_EXTENSION));

        if (in_array($extension, $this->allowedExtensions, true)) {
            return RuleResult::pass();
        }

        return RuleResult::fail(
            __('media-manager::rules.allowed_extensions.fail', [
                'extension'          => $extension,
                'allowed_extensions' => implode(', ', $this->allowedExtensions),
            ]),
            [
                'extension'          => $extension,
                'allowed_extensions' => $this->allowedExtensions,
            ]
        );
    }
}
