<?php

declare(strict_types=1);

namespace MSML\MediaManager\Models;

use Spatie\MediaLibrary\HasMedia;
use Illuminate\Database\Eloquent\Model;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

/**
 * @property int $id
 * @property int $media_namespace_id
 * @property int|null $asset_folder_id
 * @property array<string,string>|null $metadata
 * @property \Illuminate\Support\Carbon $created_at
 * @property \Illuminate\Support\Carbon $updated_at
 * @property-read MediaNamespace $namespace
 * @property-read AssetFolder|null $folder
 * @property-read string      $url
 * @property-read string      $thumb_url
 * @property-read string|null $original_name
 * @property-read string      $extension
 * @property-read string|null $mime_type
 * @property-read int|null    $size
 * @property-read string      $disk
 * @property-read string      $full_path
 * @property-read \Illuminate\Database\Eloquent\Collection<int, Model> $assetables
 *
 * @mixin \Eloquent
 */
class Asset extends Model implements HasMedia
{
    use InteractsWithMedia;

    /**
     * @var list<string>
     */
    protected $fillable = [
        'media_namespace_id',
        'asset_folder_id',
        'metadata',
    ];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'metadata' => 'array',
        ];
    }

    /**
     * @return BelongsTo<MediaNamespace, $this>
     */
    public function namespace(): BelongsTo
    {
        return $this->belongsTo(MediaNamespace::class, 'media_namespace_id');
    }

    /**
     * @return BelongsTo<AssetFolder, $this>
     */
    public function folder(): BelongsTo
    {
        return $this->belongsTo(AssetFolder::class, 'asset_folder_id');
    }

    /**
     * @return MorphToMany<Model, $this>
     */
    public function assetables(): MorphToMany
    {
        return $this->morphToMany(Model::class, 'assetable')
            ->withPivot(['collection', 'order'])
            ->withTimestamps();
    }

    public function getFirstMediaUrl(string|null $collection = null, string $conversion = ''): string
    {
        $collectionName = $collection ?? $this->namespace->name;

        return $this->getFirstMedia($collectionName)?->getUrl($conversion) ?? '';
    }

    public function getPath(): string
    {
        if ($this->folder) {
            return $this->folder->path;
        }

        return $this->namespace->slug;
    }

    /**
     * @return Attribute<string,void>
     */
    protected function url(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMediaUrl());
    }

    /**
     * @return Attribute<string,void>
     */
    protected function thumbUrl(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMediaUrl($this->namespace->name, 'thumb'));
    }

    /**
     * @return Attribute<string|null,void>
     */
    protected function originalName(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMedia($this->namespace->name)?->file_name);
    }

    /**
     * @return Attribute<string,void>
     */
    protected function extension(): Attribute
    {
        return Attribute::get(fn () => $this->original_name ? pathinfo($this->original_name, PATHINFO_EXTENSION) : '');
    }

    /**
     * @return Attribute<string|null,void>
     */
    protected function mimeType(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMedia($this->namespace->name)?->mime_type);
    }

    /**
     * @return Attribute<int|null,never>
     */
    protected function size(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMedia($this->namespace->name)?->size);
    }

    /**
     * @return Attribute<string|null,never>
     */
    protected function disk(): Attribute
    {
        return Attribute::get(fn () => $this->getFirstMedia($this->namespace->name)->disk ?? $this->namespace->disk);
    }

    /**
     * @return Attribute<non-falsy-string,void>
     */
    protected function fullPath(): Attribute
    {
        return Attribute::get(fn () => $this->folder
                ? "{$this->folder->path}/{$this->original_name}"
                : "{$this->namespace->slug}/{$this->original_name}"
        );
    }
}
