<?php

namespace MSML\Surveys\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use MSML\Surveys\Enums\SurveyStatusEnum;
use Illuminate\Database\Eloquent\Builder;
use MSML\Surveys\DataObjects\SurveySettingData;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Attributes\Scope;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * @property int $id
 * @property int $user_id
 * @property string $title
 * @property SurveySettingData $settings
 * @property Carbon|null $published_at
 * @property Carbon|null $ended_at
 * @property Carbon $created_at
 * @property Carbon $updated_at
 * @property-read bool $has_finished
 * @property-read SurveyStatusEnum::Draft|SurveyStatusEnum::Finished|SurveyStatusEnum::Published|SurveyStatusEnum::Scheduled $status
 * @property-read \Illuminate\Database\Eloquent\Collection<int, \MSML\Surveys\Models\SurveyQuestion|Model> $surveyQuestions
 * @property-read \Illuminate\Database\Eloquent\Collection<int, \MSML\Surveys\Models\SurveyQuestion|Model> $surveyResponses
 */
class Survey extends Model
{
    /**
     * @var list<string>
     */
    protected $guarded = ['id'];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'settings'     => SurveySettingData::class,
            'published_at' => 'date',
            'ended_at'     => 'date',
        ];
    }

    /**
     * @return BelongsTo<Model, $this>
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(config('auth.providers.users.model'));
    }

    /**
     * @return HasMany<SurveyQuestion|Model, $this>
     */
    public function surveyQuestions(): HasMany
    {
        return $this->hasMany(config('surveys.models.survey_question'));
    }

    /**
     * @return HasMany<SurveyResponse|Model, $this>
     */
    public function surveyResponses(): HasMany
    {
        return $this->hasMany(config('surveys.models.survey_response'));
    }

    #[Scope]
    public function global(Builder $query, string $wildcard): Builder
    {
        return $query->where(function ($query) use ($wildcard) {
            $query->where('title', 'like', '%' . $wildcard . '%');
        });
    }

    #[Scope]
    public function ofStatus(Builder $query, string $key): Builder
    {
        return $query->when($key === SurveyStatusEnum::Draft->value, function ($q) {
            $q->whereNull('published_at');
        })
            ->when($key === SurveyStatusEnum::Scheduled->value, function ($q) {
                $q->whereNotNull('published_at')->whereAfterToday('published_at');
            })
            ->when($key === SurveyStatusEnum::Published->value, function ($q) {
                $q->whereNotNull('published_at')
                    ->whereTodayOrBefore('published_at')
                    ->where(function ($query) {
                        $query->whereNull('ended_at')
                            ->orWhereTodayOrAfter('ended_at');
                    });
            })
            ->when($key === SurveyStatusEnum::Finished->value, function ($q) {
                $q->whereNotNull('published_at')
                    ->whereNotNull('ended_at')
                    ->whereBeforeToday('ended_at');
            });
    }

    /**
     * @return Attribute<SurveyStatusEnum::Draft|SurveyStatusEnum::Finished|SurveyStatusEnum::Published|SurveyStatusEnum::Scheduled,never>
     */
    public function status(): Attribute
    {
        return Attribute::get(function () {
            $now = now()->startOfDay();

            $isScheduledForFuture = $this->published_at?->gt($now) ?? false;
            $hasEnded = !is_null($this->ended_at) && $this->ended_at->lt($now);
            $isPublished = $this->published_at?->lte($now) ?? false;
            $hasNoEndDate = is_null($this->ended_at);
            $endDateInFuture = $this->ended_at?->gte($now) ?? false;

            return match (true) {
                $isScheduledForFuture                               => SurveyStatusEnum::Scheduled,
                $hasEnded                                           => SurveyStatusEnum::Finished,
                $isPublished && ($hasNoEndDate || $endDateInFuture) => SurveyStatusEnum::Published,
                default                                             => SurveyStatusEnum::Draft
            };
        });
    }

    public function publish(bool $publish = true): void
    {
        $this->update([
            'published_at' => $publish ? now() : null,
        ]);
    }
}
