<?php

declare(strict_types=1);

namespace MSML\Tables\Builders;

use Closure;
use MSML\Tables\Schema\TableSchema;
use MSML\Tables\Schema\ComputedField;
use Illuminate\Support\Traits\Macroable;
use Illuminate\Support\Traits\Conditionable;

final class TableBuilder
{
    use Conditionable, Macroable;

    /** @var array<int, ColumnBuilder> */
    private array $columns = [];

    /** @var array<int, ComputedField> */
    private array $computed = [];

    /** @var array<int, string> */
    private array $with = [];

    /** @var array<int, string> */
    private array $withCount = [];

    /** @var array<int, Closure> */
    private array $scopes = [];

    private bool $exportable = false;

    private bool $multiSort = false;

    private int $defaultPageSize = 15;

    private Closure|null $queryModifier = null;

    /**
     * @param  class-string  $model
     */
    public function __construct(
        private readonly string $model,
    ) {}

    /**
     * @param  string|array<int, string>  $relations
     */
    public function with(array|string $relations): self
    {
        $this->with = array_merge($this->with, (array)$relations);

        return $this;
    }

    /**
     * @param  string|array<int, string>  $relations
     */
    public function withCount(array|string $relations): self
    {
        $this->withCount = array_merge($this->withCount, (array)$relations);

        return $this;
    }

    public function whereNotNull(string $column): self
    {
        $this->scopes[] = fn ($query) => $query->whereNotNull($column);

        return $this;
    }

    public function where(string $column, mixed $operator = null, mixed $value = null): self
    {
        $this->scopes[] = fn ($query) => $query->where($column, $operator, $value);

        return $this;
    }

    public function whereHas(string $relation, Closure|null $callback = null): self
    {
        $this->scopes[] = fn ($query) => $query->whereHas($relation, $callback);

        return $this;
    }

    public function scope(Closure $scope): self
    {
        $this->scopes[] = $scope;

        return $this;
    }

    public function modifyQuery(Closure $modifier): self
    {
        $this->queryModifier = $modifier;

        return $this;
    }

    /**
     * @param  Closure(ColumnBuilder): ColumnBuilder  $callback
     */
    public function column(string $key, Closure $callback): self
    {
        $builder = new ColumnBuilder($key);
        $callback($builder);
        $this->columns[] = $builder;

        return $this;
    }

    public function computed(string $key, Closure|null $transformer = null, bool $exportable = false): self
    {
        $this->computed[] = new ComputedField($key, $transformer, $exportable);

        return $this;
    }

    public function exportable(bool $exportable = true): self
    {
        $this->exportable = $exportable;

        return $this;
    }

    public function multiSort(bool $enabled = true): self
    {
        $this->multiSort = $enabled;

        return $this;
    }

    public function pageSize(int $size): self
    {
        $this->defaultPageSize = $size;

        return $this;
    }

    public function compile(): TableSchema
    {
        $columns = array_map(fn (ColumnBuilder $builder) => $builder->build(), $this->columns);

        $searchableColumns = collect($this->columns)
            ->filter(fn (ColumnBuilder $builder) => $builder->isSearchable())
            ->map(fn (ColumnBuilder $builder) => $builder->getKey())
            ->values()
            ->all();

        return new TableSchema(
            model: $this->model,
            columns: $columns,
            computed: $this->computed,
            with: array_unique($this->with),
            withCount: array_unique($this->withCount),
            scopes: $this->scopes,
            searchableColumns: $searchableColumns,
            exportable: $this->exportable,
            multiSort: $this->multiSort,
            defaultPageSize: $this->defaultPageSize,
            queryModifier: $this->queryModifier,
        );
    }
}
