<?php

declare(strict_types=1);

namespace MSML\Tables\Mappers;

use MSML\Tables\Enums\FilterType;
use MSML\Tables\Enums\FormatType;
use MSML\Tables\Schema\TableSchema;
use MSML\Tables\Schema\ColumnSchema;
use MSML\Tables\Mappers\Contracts\FrontendMapper;

final class MantineReactTableMapper implements FrontendMapper
{
    public function map(TableSchema $schema, string $tableName, int $pageSize): array
    {
        return [
            'name'    => $tableName,
            'columns' => array_map(
                fn (ColumnSchema $column) => $this->mapColumn($column),
                $schema->getColumns()
            ),
            'pageSize'   => $pageSize,
            'multiSort'  => $schema->isMultiSortEnabled(),
            'exportable' => $schema->isExportable(),
        ];
    }

    private function mapColumn(ColumnSchema $column): array
    {
        $mapped = [
            'accessorKey'        => $column->key,
            'header'             => $column->header,
            'enableColumnFilter' => $column->filterable,
            'enableSorting'      => $column->sortable,
            'isVisible'          => $column->visible,
            'minSize'            => $column->minSize,
            'maxSize'            => $column->maxSize,
            'size'               => $column->size,
            'desc'               => null,
            'sortIndex'          => null,
        ];

        if ($column->filterable) {
            $mapped['filterVariant'] = $this->mapFilterType($column->filterType);

            if ($column->filterType === FilterType::DateRange) {
                $mapped['filterFn'] = 'between';
            }

            if ($column->filterOptions !== null) {
                $mapped['mantineFilterSelectProps'] = [
                    'data' => $column->filterOptions,
                ];
            }

            if ($column->filterEndpoint !== null) {
                $mapped['filterEndpoint'] = $column->filterEndpoint;
            }
        }

        if ($column->format !== FormatType::Default) {
            $mapped['formatTo'] = $column->format->value;
        }

        if ($column->route !== null) {
            $mapped['route'] = $column->route;
        }

        return $mapped;
    }

    private function mapFilterType(FilterType $type): string
    {
        return match ($type) {
            FilterType::Text      => 'text',
            FilterType::DateRange => 'date-range',
            FilterType::Select, FilterType::AsyncSelect => 'select',
            FilterType::MultiSelect, FilterType::AsyncMultiSelect => 'multi-select',
        };
    }
}
