<?php

declare(strict_types=1);

namespace MSML\Tables\Schema;

use Closure;
use MSML\Tables\Enums\FilterType;
use MSML\Tables\Enums\FormatType;
use MSML\Tables\Sorting\ScopeSort;
use MSML\Tables\Sorting\SmartSort;
use Spatie\QueryBuilder\Sorts\Sort;
use MSML\Tables\Enums\SortDirection;
use Spatie\QueryBuilder\AllowedSort;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\Filters\Filter;
use Illuminate\Database\Eloquent\Builder;

final readonly class ColumnSchema
{
    public function __construct(
        public string $key,
        public string $header,
        public bool $visible = true,
        public int|null $minSize = null,
        public int|null $maxSize = null,
        public int|null $size = null,
        public bool $searchable = false,
        public bool $filterable = false,
        public FilterType $filterType = FilterType::Text,
        public string|null $filterKey = null,
        public array|null $filterOptions = null,
        public string|null $filterEndpoint = null,
        public string|null $filterScope = null,
        public Closure|Filter|null $customFilter = null,
        public bool $sortable = false,
        public SortDirection|null $defaultSort = null,
        public string|null $sortKey = null,
        public string|null $sortScope = null,
        public Closure|Sort|null $customSort = null,
        public FormatType $format = FormatType::Default,
        public string|null $route = null,
        public Closure|null $resourceTransformer = null,
        public Closure|null $exportTransformer = null,
    ) {}

    public function getAccessorKey(): string
    {
        return $this->key;
    }

    public function getFilterKey(): string
    {
        return $this->filterKey ?? $this->key;
    }

    public function getSortKey(): string
    {
        return $this->sortKey ?? $this->key;
    }

    public function isRelation(): bool
    {
        return str_contains($this->key, '.');
    }

    public function getRelationName(): string|null
    {
        if (!$this->isRelation()) {
            return null;
        }

        $parts = explode('.', $this->key);
        array_pop($parts);

        return implode('.', $parts);
    }

    public function getAttributeName(): string
    {
        if (!$this->isRelation()) {
            return $this->key;
        }

        $parts = explode('.', $this->key);

        return end($parts);
    }

    public function buildAllowedFilter(): AllowedFilter|null
    {
        if (!$this->filterable) {
            return null;
        }

        $filterKey = $this->getFilterKey();

        if ($this->customFilter !== null) {
            if ($this->customFilter instanceof Closure) {
                return AllowedFilter::callback($this->key, $this->customFilter);
            }

            return AllowedFilter::custom($this->key, $this->customFilter, $filterKey);
        }

        if ($this->filterScope !== null) {
            return AllowedFilter::scope($this->key, $this->filterScope);
        }

        if ($this->filterEndpoint !== null) {
            $relation = $this->getRelationNameFromKey($filterKey);

            if ($relation !== null) {
                return AllowedFilter::callback($this->key, function (Builder $query, $value) use ($relation) {
                    $ids = collect($value)->flatten()->all();

                    $query->whereHas($relation, fn (Builder $q) => $q->whereIn('id', $ids));
                });
            }
        }

        return AllowedFilter::partial($this->key, $filterKey);
    }

    public function buildAllowedSort(): AllowedSort|null
    {
        if (!$this->sortable) {
            return null;
        }

        $sortKey = $this->getSortKey();

        if ($this->customSort !== null) {
            if ($this->customSort instanceof Closure) {
                return AllowedSort::callback($this->key, $this->customSort);
            }

            return AllowedSort::custom($this->key, $this->customSort, $sortKey);
        }

        if ($this->sortScope !== null) {
            return AllowedSort::custom($this->key, new ScopeSort($this->sortScope), $sortKey);
        }

        return AllowedSort::custom($this->key, new SmartSort, $sortKey);
    }

    public function transformValue(mixed $item): mixed
    {
        if ($this->resourceTransformer !== null) {
            return ($this->resourceTransformer)($item);
        }

        return data_get($item, $this->key);
    }

    public function transformExportValue(mixed $item): mixed
    {
        if ($this->exportTransformer !== null) {
            return ($this->exportTransformer)($item);
        }

        return $this->transformValue($item);
    }

    private function getRelationNameFromKey(string $key): string|null
    {
        if (!str_contains($key, '.')) {
            return null;
        }

        $parts = explode('.', $key);
        array_pop($parts);

        return implode('.', $parts);
    }
}
