<?php

declare(strict_types=1);

namespace MSML\Tables\Schema;

use Closure;
use Illuminate\Http\Request;
use MSML\Tables\Enums\SortDirection;
use Spatie\QueryBuilder\QueryBuilder;
use Illuminate\Database\Eloquent\Builder;

final class TableSchema
{
    /**
     * @param  class-string  $model
     * @param  array<int, ColumnSchema>  $columns
     * @param  array<int, ComputedField>  $computed
     * @param  array<int, string>  $with
     * @param  array<int, string>  $withCount
     * @param  array<int, Closure>  $scopes
     * @param  array<int, string>  $searchableColumns
     */
    public function __construct(
        private string $model,
        private array $columns = [],
        private array $computed = [],
        private array $with = [],
        private array $withCount = [],
        private array $scopes = [],
        private array $searchableColumns = [],
        private bool $exportable = false,
        private bool $multiSort = false,
        private int $defaultPageSize = 15,
        private Closure|null $queryModifier = null,
    ) {}

    public function getModel(): string
    {
        return $this->model;
    }

    /**
     * @return array<int, ColumnSchema>
     */
    public function getColumns(): array
    {
        return $this->columns;
    }

    /**
     * @return array<int, ComputedField>
     */
    public function getComputed(): array
    {
        return $this->computed;
    }

    public function isExportable(): bool
    {
        return $this->exportable;
    }

    public function isMultiSortEnabled(): bool
    {
        return $this->multiSort;
    }

    public function getDefaultPageSize(): int
    {
        return $this->defaultPageSize;
    }

    public function getSearchableColumns(): array
    {
        return $this->searchableColumns;
    }

    public function query(Request $request): Builder
    {
        $query = QueryBuilder::for($this->model, $request)
            ->allowedFilters($this->buildAllowedFilters())
            ->allowedSorts($this->buildAllowedSorts());

        $defaultSort = $this->getDefaultSort();
        if ($defaultSort !== null) {
            $query->defaultSort($defaultSort);
        }

        foreach ($this->scopes as $scope) {
            $query->tap($scope);
        }

        if ($this->queryModifier !== null) {
            ($this->queryModifier)($query);
        }

        // Global exclude_ids support for picker tables
        $this->applyExcludeIds($query, $request);

        return $query->getEloquentBuilder()
            ->with($this->with)
            ->withCount($this->withCount);
    }

    private function applyExcludeIds(QueryBuilder $query, Request $request): void
    {
        $excludeIds = $request->input('custom.exclude_ids');

        if (empty($excludeIds)) {
            return;
        }

        $ids = array_filter(explode(',', $excludeIds));

        if (!empty($ids)) {
            $table = (new $this->model)->getTable();
            $query->whereNotIn("{$table}.id", $ids);
        }
    }

    public function resource(mixed $item): array
    {
        $data = [];

        foreach ($this->columns as $column) {
            $this->setNestedValue($data, $column->key, $column->transformValue($item));
        }

        foreach ($this->computed as $field) {
            $this->setNestedValue($data, $field->key, $field->transformValue($item));
        }

        return $data;
    }

    private function buildAllowedFilters(): array
    {
        $filters = [];

        $searchableColumns = $this->searchableColumns;
        $filters[] = \Spatie\QueryBuilder\AllowedFilter::callback('global', function (Builder $query, $value) use ($searchableColumns) {
            if (empty($searchableColumns) || empty($value)) {
                return;
            }

            $query->where(function (Builder $q) use ($value, $searchableColumns) {
                foreach ($searchableColumns as $column) {
                    if (str_contains($column, '.')) {
                        $parts = explode('.', $column);
                        $field = array_pop($parts);
                        $relation = implode('.', $parts);

                        $q->orWhereHas($relation, function (Builder $relationQuery) use ($field, $value) {
                            $relationQuery->where($field, 'ILIKE', "%{$value}%");
                        });
                    } else {
                        $q->orWhere($q->qualifyColumn($column), 'ILIKE', "%{$value}%");
                    }
                }
            });
        });

        foreach ($this->columns as $column) {
            $filter = $column->buildAllowedFilter();
            if ($filter !== null) {
                $filters[] = $filter;
            }
        }

        return $filters;
    }

    private function buildAllowedSorts(): array
    {
        $sorts = [];

        foreach ($this->columns as $column) {
            $sort = $column->buildAllowedSort();
            if ($sort !== null) {
                $sorts[] = $sort;
            }
        }

        return $sorts;
    }

    private function getDefaultSort(): string|null
    {
        foreach ($this->columns as $column) {
            if ($column->defaultSort !== null) {
                $prefix = $column->defaultSort === SortDirection::Desc ? '-' : '';

                return $prefix . $column->key;
            }
        }

        return null;
    }

    private function setNestedValue(array &$array, string $key, mixed $value): void
    {
        $keys = explode('.', $key);
        $current = &$array;

        foreach ($keys as $i => $k) {
            if ($i === count($keys) - 1) {
                $current[$k] = $value;
            } else {
                if (!isset($current[$k]) || !is_array($current[$k])) {
                    $current[$k] = [];
                }
                $current = &$current[$k];
            }
        }
    }
}
