<?php

declare(strict_types=1);

namespace MSML\Tables;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use MSML\Tables\Schema\TableSchema;
use Illuminate\Database\Eloquent\Builder;
use MSML\Tables\Mappers\Contracts\FrontendMapper;

abstract class Table
{
    protected static string|null $name = null;

    private TableSchema|null $schema = null;

    private FrontendMapper|null $mapper = null;

    abstract public function authorize(): bool;

    abstract public function build(): TableSchema;

    public function getSchema(): TableSchema
    {
        return $this->schema ??= $this->build();
    }

    public function query(Request $request): Builder
    {
        return $this->getSchema()->query($request);
    }

    public function resource(mixed $item): array
    {
        return $this->getSchema()->resource($item);
    }

    public function isExportable(): bool
    {
        return $this->getSchema()->isExportable();
    }

    public function getExportChunkSize(): int
    {
        return 2500;
    }

    public function toArray(): array
    {
        return $this->getMapper()->map(
            $this->getSchema(),
            static::name(),
            $this->getPageSize()
        );
    }

    protected function getPageSize(): int
    {
        return $this->getSchema()->getDefaultPageSize();
    }

    protected function getMapper(): FrontendMapper
    {
        return $this->mapper ??= app(FrontendMapper::class);
    }

    public static function name(): string
    {
        if (static::$name !== null) {
            return static::$name;
        }

        $baseNamespace = config('tables.table_class_namespace', 'App\\Tables');
        $fullClass = static::class;

        $relativeName = str_replace($baseNamespace . '\\', '', $fullClass);

        $parts = explode('\\', $relativeName);
        $parts = array_map(fn ($part) => Str::snake($part, '-'), $parts);

        return implode('/', $parts);
    }
}
