<?php

declare(strict_types=1);

namespace MSML\Tables\Traits;

use Generator;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Builder;

trait ExportGenerator
{
    public function withExport(Builder $query): Builder
    {
        return $query;
    }

    public function formatExportableColumn(mixed $item, string|null $formatTo, string $accessorKey): mixed
    {
        $value = data_get($item, $accessorKey);
        $parser = data_get($this->customExportableColumnParser(), $accessorKey);

        return match ($formatTo) {
            'date'     => Carbon::make($value)?->format('d-m-Y'),
            'datetime' => Carbon::make($value)?->format('d-m-Y H:i:s'),
            'custom'   => is_callable($parser) ? $parser($item) : $value,
            default    => $value
        };
    }

    /**
     * @return array<string, callable>
     */
    public function customExportableColumnParser(): array
    {
        return [];
    }

    /**
     * @return array<string, mixed>
     */
    public function customExportableColumns(mixed $item): array
    {
        return [];
    }

    public function generator(Request $request, Collection $columns): Generator
    {
        $tableQuery = $this->query($request)->pipe($this->withExport(...));

        $tableResults = $tableQuery->orderByDesc('id')->lazy(2500);

        foreach ($tableResults as $item) {
            $values = $columns->mapWithKeys(fn ($column) => [
                $column['header'] => $this->formatExportableColumn($item, data_get($column, 'formatTo'), data_get($column, 'accessorKey')),
            ]);

            yield $values->merge($this->customExportableColumns($item))->toArray();
        }
    }
}
