<?php

namespace MSML\Tables\Commands;

use Illuminate\Console\Command;
use Illuminate\Filesystem\Filesystem;
use InvalidArgumentException;

class MakeTableCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'make:table {name : The name of the table class} {--group=default : The group name} {--path= : Path to write the table class file to}';

    /**
     * The console command name.
     *
     * @var string
     */

    protected $name = 'make:table';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create a new Table Class';

    /**
     * @var Filesystem
     */
    protected Filesystem $files;

    public function __construct(Filesystem $files)
    {
        parent::__construct();

        $this->files = $files;
    }

    public function handle()
    {
        $name = trim($this->input->getArgument('name'));
        $path = trim($this->input->getOption('path'));

        if (empty($path)) {
            $path = $this->resolveTablesPath();
        }

        $this->ensureTableClassDoesntAlreadyExist($name, $path);

        $this->files->ensureDirectoryExists($path);

        $this->files->put(
            $this->getPath($name, $path),
            $this->getContent($name, $path)
        );
    }

    protected function getStub(): string
    {
        return <<<'EOT'
<?php

namespace {{ namespace }};

use MSML\Tables\Table;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use MSML\Tables\Interfaces\Exportable;
use MSML\Tables\Traits\ExportGenerator;
use Illuminate\Database\Eloquent\Builder;

class {{ class }} extends Table implements Exportable
{
    use ExportGenerator;

    public function authorize(): bool
    {
        return Gate::allows('viewAny', Model::class);
    }

    public function query(Request $request): Builder
    {
        return Model::query();
    }

    public function resource(mixed $item): array
    {
        return [
            'id'         => $item->id,
            'created_at' => $item->created_at,
        ];
    }

    protected function defineColumns(): array
    {
        return [
            [
                'accessorKey'        => 'id',
                'header'             => 'Identifier',
                'filterVariant'      => static::filterVariants()->text(),
                'enableColumnFilter' => true,
                'enableSorting'      => false,

                // Own props
                'isVisible' => true,
            ],
            [
                'accessorKey'        => 'user.full_name',
                'id'                 => 'created_by',
                'header'             => 'Created by',
                'filterVariant'      => static::filterVariants()->text(),
                'enableColumnFilter' => true,
                'enableSorting'      => false,

                // Own props
                'isVisible' => true,
            ],
            [
                'accessorKey'        => 'created_at',
                'header'             => 'Last modified',
                'filterVariant'      => static::filterVariants()->dateRange(),
                'filterFn'           => static::filterFunctions()->between(),
                'enableSorting'      => true,
                'enableColumnFilter' => true,
                'desc'               => true,

                // Own props
                'isVisible' => true,
                'formatTo'  => static::formatTo()->date(),
            ],
        ];
    }

    public function customExportableColumnParser(): array
    {
        return [
            'created_by' => fn ($item) => $item->user->id,
        ];
    }
}
EOT;
    }

    protected function getContent($name, $path)
    {
        return str_replace(
            ['{{ namespace }}', '{{ class }}'],
            [$this->getNamespace($path), $name],
            $this->getStub()
        );
    }

    protected function ensureTableClassDoesntAlreadyExist($name, $path): void
    {
        if ($this->files->exists($this->getPath($name, $path))) {
            throw new InvalidArgumentException(sprintf('%s already exists!', $name));
        }
    }

    protected function resolveTablesPath(): string
    {
        return config('tables.table_class_path', app_path('Tables'));
    }

    protected function getPath($name, $path): string
    {
        return $path . '/' . $name . '.php';
    }

    protected function getNamespace($path): string
    {
        $path = preg_replace(
            [
                '/^(' . preg_quote(base_path(), '/') . ')/',
                '/\//',
            ],
            [
                '',
                '\\',
            ],
            $path
        );

        $namespace = implode('\\', array_map(fn ($directory) => ucfirst($directory), explode('\\', $path)));

        // Remove leading backslash if present
        if (substr($namespace, 0, 1) === '\\') {
            $namespace = substr($namespace, 1);
        }

        return $namespace;
    }
}
