# Laravel Tables

A Laravel package for building configurable data tables with a fluent DSL.

## Documentation

### Getting Started
1. [Introduction](docs/01-introduction.md) - Package overview and core concepts
2. [Getting Started](docs/02-getting-started.md) - Installation and your first table

### Core Features
3. [Columns](docs/03-columns.md) - Column configuration, formatting, and transformers
4. [Filtering](docs/04-filtering.md) - Text, select, async, and custom filters
5. [Sorting](docs/05-sorting.md) - Smart sorting, relations, and custom sorts

### Advanced Usage
6. [Exporting](docs/06-exporting.md) - Export configuration and custom export logic
7. [Advanced Patterns](docs/07-advanced-patterns.md) - Macros, computed fields, and reusable patterns
8. [Relational Tables](docs/08-relational-tables.md) - Many-to-many relationships with pivot data

## Package Architecture

Laravel Tables follows a **Compiler-Based DSL** architecture:
```
TableCompiler → ColumnBuilder → TableSchema → FrontendMapper
```

For relational tables:
```
RelationalTable → PivotColumnBuilder → PivotColumnSchema
                                    ↘ picker_table config
```

## Quick Example

### Standard Table
```php
<?php

namespace App\Tables\Users;

use App\Tables\BaseTable;
use App\Models\User;
use MSML\Tables\TableCompiler;
use MSML\Tables\Enums\FormatType;
use MSML\Tables\Schema\TableSchema;
use Illuminate\Support\Facades\Gate;

class UsersTable extends BaseTable
{
    public function authorize(): bool
    {
        return Gate::allows('viewAny', User::class);
    }

    public function build(): TableSchema
    {
        return TableCompiler::for(User::class)
            ->with(['role'])

            ->column('name', fn ($col) => $col
                ->header('Name')
                ->searchable()
                ->filterable()
                ->sortable()
                ->format(FormatType::Link)
                ->route('users/{id}/edit')
            )

            ->column('role.name', fn ($col) => $col
                ->header('Role')
                ->filterable(endpoint: '/api/common/roles')
                ->sortable()
            )

            ->column('status', fn ($col) => $col
                ->header('Status')
                ->filterable(options: StatusEnum::toSelectOptions())
                ->sortable()
                ->format(FormatType::Status)
                ->resource(fn ($item) => [
                    'name' => $item->status->name(),
                    'color' => $item->status->color(),
                ])
                ->export(fn ($item) => $item->status->name())
            )

            ->auditColumns()
            ->resourceComputed()
            ->exportable()
            ->compile();
    }
}
```

### Relational Table
```php
<?php

namespace App\Tables\Assignments;

use App\Models\Product;
use App\Tables\RelationalTable;
use App\Tables\Products\ProductsTable as BaseProductsTable;
use MSML\Tables\Builders\TableBuilder;
use Illuminate\Support\Facades\Gate;

class AssignmentProductsTable extends RelationalTable
{
    protected string $model = Product::class;
    protected string $relation = 'assignments';
    protected string $parentKey = 'assignment_id';
    protected string $includeKey = 'include_product_ids';
    protected string $excludeKey = 'exclude_product_ids';

    protected ?string $pickerTable = BaseProductsTable::class;

    public function authorize(): bool
    {
        return Gate::allows('viewAny', Product::class);
    }

    protected function pivotColumns(): void
    {
        $this->pivotColumn('minimum', fn ($col) => $col
            ->header('Minimum')
            ->number()
            ->default(0)
        );

        $this->pivotColumn('adjustable', fn ($col) => $col
            ->header('Adjustable')
            ->switch()
            ->default(false)
        );
    }

    protected function columns(TableBuilder $builder): TableBuilder
    {
        return $builder
            ->column('name', fn ($col) => $col->header('Name'))
            ->column('external_id', fn ($col) => $col->header('ID'));
    }
}
```