<?php

declare(strict_types=1);

namespace MSML\Tables\Builders;

use Closure;
use MSML\Tables\Enums\FilterType;
use MSML\Tables\Enums\FormatType;
use MSML\Tables\Enums\SortDirection;
use MSML\Tables\Schema\ColumnSchema;

final class ColumnBuilder
{
    private string $header;

    private bool $visible = true;

    private int|null $minSize = null;

    private int|null $maxSize = null;

    private int|null $size = null;

    private bool $searchable = false;

    private bool $filterable = false;

    private FilterType $filterType = FilterType::Text;

    private string|null $filterKey = null;

    private array|null $filterOptions = null;

    private string|null $filterEndpoint = null;

    private string|null $filterScope = null;

    private mixed $customFilter = null;

    private bool $sortable = false;

    private SortDirection|null $defaultSort = null;

    private string|null $sortKey = null;

    private string|null $sortScope = null;

    private mixed $customSort = null;

    private FormatType $format = FormatType::Default;

    private string|null $route = null;

    private Closure|null $resourceTransformer = null;

    private Closure|null $exportTransformer = null;

    public function __construct(
        private readonly string $key,
    ) {
        $this->header = $this->generateHeaderFromKey($key);
    }

    public function header(string $header): self
    {
        $this->header = $header;

        return $this;
    }

    public function visible(bool $visible = true): self
    {
        $this->visible = $visible;

        return $this;
    }

    public function hidden(): self
    {
        $this->visible = false;

        return $this;
    }

    public function minSize(int|null $size = null): self
    {
        $this->minSize = $size;

        return $this;
    }

    public function maxSize(int|null $size = null): self
    {
        $this->maxSize = $size;

        return $this;
    }

    public function size(int|null $size = null, int|null $min = null, int|null $max = null): self
    {
        $this->size = $size;

        $this->minSize($min);
        $this->maxSize($max);

        return $this;
    }

    public function searchable(bool $searchable = true): self
    {
        $this->searchable = $searchable;

        return $this;
    }

    public function filterable(
        string|null $key = null,
        FilterType|null $type = null,
        array|null $options = null,
        string|null $endpoint = null,
        string|null $scope = null,
    ): self {
        $this->filterable = true;
        $this->filterKey = $key;
        $this->filterOptions = $options;
        $this->filterEndpoint = $endpoint;
        $this->filterScope = $scope;
        $this->filterType = $type ?? $this->inferFilterType($options, $endpoint);

        return $this;
    }

    public function customFilter(mixed $filter): self
    {
        $this->filterable = true;
        $this->customFilter = $filter;

        return $this;
    }

    public function sortable(string|null $key = null, string|null $scope = null): self
    {
        $this->sortable = true;
        $this->sortKey = $key;
        $this->sortScope = $scope;

        return $this;
    }

    public function defaultSort(SortDirection|string $direction = SortDirection::Asc): self
    {
        $this->sortable = true;
        $this->defaultSort = $direction instanceof SortDirection
            ? $direction
            : SortDirection::from($direction);

        return $this;
    }

    public function customSort(mixed $sort): self
    {
        $this->sortable = true;
        $this->customSort = $sort;

        return $this;
    }

    public function format(FormatType|string $format): self
    {
        $this->format = $format instanceof FormatType ? $format : FormatType::from($format);

        return $this;
    }

    public function route(string $route): self
    {
        $this->route = $route;

        return $this;
    }

    public function resource(Closure $transformer): self
    {
        $this->resourceTransformer = $transformer;

        return $this;
    }

    public function export(Closure $transformer): self
    {
        $this->exportTransformer = $transformer;

        return $this;
    }

    public function build(): ColumnSchema
    {
        return new ColumnSchema(
            key: $this->key,
            header: $this->header,
            visible: $this->visible,
            minSize: $this->minSize,
            maxSize: $this->maxSize,
            size: $this->size,
            searchable: $this->searchable,
            filterable: $this->filterable,
            filterType: $this->filterType,
            filterKey: $this->filterKey,
            filterOptions: $this->filterOptions,
            filterEndpoint: $this->filterEndpoint,
            filterScope: $this->filterScope,
            customFilter: $this->customFilter,
            sortable: $this->sortable,
            defaultSort: $this->defaultSort,
            sortKey: $this->sortKey,
            sortScope: $this->sortScope,
            customSort: $this->customSort,
            format: $this->format,
            route: $this->route,
            resourceTransformer: $this->resourceTransformer,
            exportTransformer: $this->exportTransformer,
        );
    }

    public function getKey(): string
    {
        return $this->key;
    }

    public function isSearchable(): bool
    {
        return $this->searchable;
    }

    private function generateHeaderFromKey(string $key): string
    {
        $parts = explode('.', $key);
        $lastPart = end($parts);

        return ucfirst(str_replace('_', ' ', $lastPart));
    }

    private function inferFilterType(array|null $options, string|null $endpoint): FilterType
    {
        if ($endpoint !== null) {
            return FilterType::AsyncMultiSelect;
        }

        if ($options !== null) {
            return FilterType::MultiSelect;
        }

        return FilterType::Text;
    }
}
