<?php

declare(strict_types=1);

namespace MSML\Tables\Builders;

use MSML\Tables\Enums\PivotInputType;
use MSML\Tables\Schema\PivotColumnSchema;

final class PivotColumnBuilder
{
    private string $header;

    private PivotInputType $type = PivotInputType::Number;

    private mixed $default = null;

    private array $options = [];

    private bool $editable = true;

    public function __construct(
        private readonly string $name,
    ) {
        $this->header = ucfirst(str_replace('_', ' ', $name));
    }

    public function header(string $header): self
    {
        $this->header = $header;

        return $this;
    }

    public function type(PivotInputType $type): self
    {
        $this->type = $type;

        return $this;
    }

    public function number(): self
    {
        $this->type = PivotInputType::Number;

        return $this;
    }

    public function switch(): self
    {
        $this->type = PivotInputType::Switch;

        return $this;
    }

    public function text(): self
    {
        $this->type = PivotInputType::Text;

        return $this;
    }

    /**
     * @param  array<int, array{value: string, label: string}>  $options
     */
    public function select(array $options = []): self
    {
        $this->type = PivotInputType::Select;
        $this->options = $options;

        return $this;
    }

    public function default(mixed $value): self
    {
        $this->default = $value;

        return $this;
    }

    /**
     * @param  array<int, array{value: string, label: string}>  $options
     */
    public function options(array $options): self
    {
        $this->options = $options;

        return $this;
    }

    public function editable(bool $editable = true): self
    {
        $this->editable = $editable;

        return $this;
    }

    public function readOnly(): self
    {
        $this->editable = false;

        return $this;
    }

    public function build(): PivotColumnSchema
    {
        return new PivotColumnSchema(
            name: $this->name,
            header: $this->header,
            type: $this->type,
            default: $this->default,
            options: $this->options,
            editable: $this->editable,
        );
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function getHeader(): string
    {
        return $this->header;
    }

    public function getType(): PivotInputType
    {
        return $this->type;
    }

    public function getDefault(): mixed
    {
        return $this->default;
    }
}
