<?php

declare(strict_types=1);

namespace MSML\Tables\Commands;

use Illuminate\Support\Str;
use Illuminate\Console\Command;
use Illuminate\Filesystem\Filesystem;

class MakeTableCommand extends Command
{
    protected $signature = 'make:table {name : The name of the table class (e.g. Users/UsersTable)}';

    protected $description = 'Create a new Table class';

    public function __construct(private Filesystem $files)
    {
        parent::__construct();
    }

    public function handle(): int
    {
        $name = str_replace('/', '\\', trim($this->argument('name')));

        $path = $this->getPath($name);

        if ($this->files->exists($path)) {
            $this->error("Table [{$name}] already exists!");

            return self::FAILURE;
        }

        $this->files->ensureDirectoryExists(dirname($path));

        $this->files->put($path, $this->buildClass($name));

        $this->info("Table [{$path}] created successfully.");

        return self::SUCCESS;
    }

    private function getPath(string $name): string
    {
        $basePath = config('tables.table_class_path', app_path('Tables'));

        return $basePath . '/' . str_replace('\\', '/', $name) . '.php';
    }

    private function buildClass(string $name): string
    {
        $namespace = $this->getNamespace($name);
        $class = $this->getClassName($name);
        $model = $this->guessModelName($class);
        $route = $this->guessRouteName($class);

        return str_replace(
            ['{{ namespace }}', '{{ class }}', '{{ model }}', '{{ route }}'],
            [$namespace, $class, $model, $route],
            $this->getStub()
        );
    }

    private function guessRouteName(string $class): string
    {
        $name = Str::beforeLast($class, 'Table');

        return Str::kebab(Str::plural($name));
    }

    private function getNamespace(string $name): string
    {
        $baseNamespace = config('tables.table_class_namespace', 'App\\Tables');

        if (!Str::contains($name, '\\')) {
            return $baseNamespace;
        }

        return $baseNamespace . '\\' . Str::beforeLast($name, '\\');
    }

    private function getClassName(string $name): string
    {
        return Str::afterLast($name, '\\');
    }

    private function guessModelName(string $class): string
    {
        $name = Str::beforeLast($class, 'Table');

        return Str::singular($name);
    }

    private function getStub(): string
    {
        return <<<'PHP'
<?php

namespace {{ namespace }};

use App\Tables\BaseTable;
use App\Models\{{ model }};
use MSML\Tables\TableCompiler;
use MSML\Tables\Enums\FilterType;
use MSML\Tables\Enums\FormatType;
use MSML\Tables\Schema\TableSchema;
use Illuminate\Support\Facades\Gate;
use Illuminate\Database\Eloquent\Builder;

class {{ class }} extends BaseTable
{
    public function authorize(): bool
    {
        return Gate::allows('viewAny', {{ model }}::class);
    }

    public function build(): TableSchema
    {
        return TableCompiler::for({{ model }}::class)
            // Eager load relations
            // ->with(['relation'])
            // ->withCount(['relation'])

            // Apply scopes
            // ->scope(fn ($query) => $query->active())

            // Modify query
            // ->modifyQuery(function (Builder $query) {
            //     $query->where('column', 'value');
            // })

            // Columns
            ->column('name', fn ($col) => $col
                ->header('Name')
                ->searchable()
                ->filterable()
                ->sortable()
                ->format(FormatType::Link)
                ->route('{{ route }}/{id}/edit')
            )

            // Relation column
            // ->column('relation.name', fn ($col) => $col
            //     ->header('Relation')
            //     ->filterable(endpoint: '/api/common/relations')
            //     ->sortable()
            // )

            // Status column with enum
            // ->column('status', fn ($col) => $col
            //     ->header('Status')
            //     ->filterable(options: array_map(fn ($e) => [
            //         'value' => (string) $e->value,
            //         'label' => $e->name(),
            //     ], StatusEnum::cases()))
            //     ->sortable()
            //     ->format(FormatType::Status)
            //     ->resource(fn ($item) => [
            //         'name' => $item->status->name(),
            //         'color' => $item->status->color(),
            //     ])
            // )

            // Date column
            // ->column('date', fn ($col) => $col
            //     ->header('Date')
            //     ->filterable(type: FilterType::DateRange)
            //     ->customFilter(new FiltersDateRange)
            //     ->sortable()
            //     ->format(FormatType::Date)
            // )

            // Hidden column
            // ->column('hidden_column', fn ($col) => $col
            //     ->header('Hidden')
            //     ->hidden()
            // )

            // Custom export output
            // ->column('formatted', fn ($col) => $col
            //     ->header('Formatted')
            //     ->resource(fn ($item) => ['complex' => 'data'])
            //     ->export(fn ($item) => $item->simple_value)
            // )

            ->auditColumns()
            ->resourceComputed()

            // Options
            // ->exportable()
            // ->multiSort()
            // ->pageSize(25)

            ->compile();
    }

    // Optional: Add dynamic columns to export
    // public function customExportableColumns(mixed $item): array
    // {
    //     return [
    //         'Dynamic Column' => $item->computed_value,
    //     ];
    // }

    // Optional: Custom export filename
    // public function getFilename(Request $request): string
    // {
    //     return 'custom-name-' . now()->format('Y-m-d') . '.xlsx';
    // }
}
PHP;
    }
}
