<?php

declare(strict_types=1);

namespace MSML\Tables;

use Closure;
use MSML\Tables\Enums\FormatType;
use MSML\Tables\Schema\TableSchema;
use MSML\Tables\Builders\TableBuilder;
use MSML\Tables\Schema\PivotColumnSchema;
use MSML\Tables\Builders\PivotColumnBuilder;

abstract class RelationalTable extends Table
{
    /** @var class-string */
    protected string $model;

    protected string $relation = 'relation';

    protected string $parentKey = 'parent_id';

    protected string $includeKey = 'include_ids';

    protected string $excludeKey = 'exclude_ids';

    /** @var class-string<Table>|null */
    protected string|null $pickerTable = null;

    /** @var array<string, PivotColumnBuilder> */
    private array $pivotColumns = [];

    abstract protected function columns(TableBuilder $builder): TableBuilder;

    protected function pivotColumns(): void
    {
        // Override in child classes to define pivot columns
    }

    /**
     * @param  Closure(PivotColumnBuilder): self  $callback
     */
    public function pivotColumn(string $name, Closure|null $callback = null): self
    {
        $builder = new PivotColumnBuilder($name);

        if ($callback !== null) {
            $callback($builder);
        }

        $this->pivotColumns[$name] = $builder;

        return $this;
    }

    public function build(): TableSchema
    {
        $this->pivotColumns();

        $builder = TableCompiler::for($this->model)
            ->modifyQuery(fn ($query) => $this->applyRelationalQuery($query));

        $builder = $this->columns($builder);
        $builder = $this->applyPivotColumnsToBuilder($builder);

        return $builder->compile();
    }

    protected function applyRelationalQuery(mixed $query): void
    {
        $request = request();

        $parentId = $request->input("custom.{$this->parentKey}");
        $includeIds = array_filter(explode(',', $request->input("custom.{$this->includeKey}", '') ?? ''));
        $excludeIds = array_filter(explode(',', $request->input("custom.{$this->excludeKey}", '') ?? ''));

        $tableName = (new $this->model)->getTable();

        $query->where(function ($q) use ($parentId, $includeIds, $tableName) {
            $q->withWhereHas($this->relation, fn ($sub) => $sub->where("{$this->relation}.id", $parentId))
                ->orWhereIn("{$tableName}.id", $includeIds);
        })->whereNotIn("{$tableName}.id", $excludeIds);
    }

    protected function applyPivotColumnsToBuilder(TableBuilder $builder): TableBuilder
    {
        foreach ($this->pivotColumns as $name => $pivotBuilder) {
            $builder->column("pivot.{$name}", fn ($col) => $col
                ->header($pivotBuilder->getHeader())
                ->format(FormatType::Custom)
                ->resource(fn ($item) => $item->{$this->relation}->value("pivot.{$name}") ?? $pivotBuilder->getDefault())
            );
        }

        // Allow child classes to add computed fields
        $this->computedFields($builder);

        // Always include ID
        $builder->computed('id');

        return $builder;
    }

    /**
     * Override to add additional computed fields.
     */
    protected function computedFields(TableBuilder $builder): void
    {
        // Override in child classes
    }

    /**
     * @return array<string, PivotColumnSchema>
     */
    public function getPivotConfig(): array
    {
        $this->pivotColumns();

        return collect($this->pivotColumns)
            ->mapWithKeys(fn (PivotColumnBuilder $builder, string $name) => [
                $name => $builder->build(),
            ])
            ->all();
    }

    public function getRelationConfig(): array
    {
        return [
            'parent_key'  => $this->parentKey,
            'include_key' => $this->includeKey,
            'exclude_key' => $this->excludeKey,
        ];
    }

    public function getPickerTable(): array|null
    {
        if ($this->pickerTable === null) {
            return null;
        }

        $table = new $this->pickerTable;

        return $table->toArray();
    }

    public function toArray(): array
    {
        return parent::toArray() + [
            'pivot_config' => collect($this->getPivotConfig())
                ->map(fn (PivotColumnSchema $schema) => $schema->toArray())
                ->all(),
            'relation_config' => $this->getRelationConfig(),
            'picker_table'    => $this->getPickerTable(),
        ];
    }
}
