<?php

declare(strict_types=1);

namespace MSML\Tables\Sorting;

use Illuminate\Support\Str;
use Spatie\QueryBuilder\Sorts\Sort;
use Illuminate\Database\Eloquent\Builder;

final class SmartSort implements Sort
{
    public function __invoke(Builder $query, bool $descending, string $property): Builder
    {
        if (str_contains($property, '.')) {
            return $this->sortByRelation($query, $descending, $property);
        }

        return $this->sortByColumn($query, $descending, $property);
    }

    private function sortByColumn(Builder $query, bool $descending, string $property): Builder
    {
        $direction = $descending ? 'DESC' : 'ASC';
        $nulls = $descending ? 'LAST' : 'FIRST';
        $column = $query->qualifyColumn($property);

        return $query->orderByRaw("$column $direction NULLS $nulls");
    }

    private function sortByRelation(Builder $query, bool $descending, string $property): Builder
    {
        $direction = $descending ? 'DESC' : 'ASC';

        $relation = Str::beforeLast($property, '.');
        $column = Str::afterLast($property, '.');

        $model = $query->getModel();

        if ($this->isSelfReferencingRelation($model, $relation)) {
            return $this->sortBySelfRelation($query, $descending, $relation, $column);
        }

        $qualifiedColumn = $this->getRelatedTableColumn($model, $relation, $column);

        $query->leftJoinRelationship($relation);

        return $query->orderByRaw("$qualifiedColumn $direction NULLS LAST");
    }

    private function getRelatedTableColumn(object $model, string $relation, string $column): string
    {
        $current = $model;

        foreach (explode('.', $relation) as $segment) {
            if (!method_exists($current, $segment)) {
                throw new \InvalidArgumentException("Relation {$segment} does not exist on " . get_class($current));
            }
            $current = $current->{$segment}()->getRelated();
        }

        return $current->getTable() . '.' . $column;
    }

    private function isSelfReferencingRelation(object $model, string $relation): bool
    {
        $segments = explode('.', $relation);
        $current = $model;

        foreach (array_slice($segments, 0, -1) as $segment) {
            if (!method_exists($current, $segment)) {
                return false;
            }
            $current = $current->{$segment}()->getRelated();
        }

        $lastSegment = end($segments);

        if (!method_exists($current, $lastSegment)) {
            return false;
        }

        $relationInstance = $current->{$lastSegment}();

        if (!method_exists($relationInstance, 'getRelated')) {
            return false;
        }

        return $current->getTable() === $relationInstance->getRelated()->getTable();
    }

    private function sortBySelfRelation(
        Builder $query,
        bool $descending,
        string $relation,
        string $column
    ): Builder {
        $direction = $descending ? 'DESC' : 'ASC';
        $alias = Str::snake(Str::replace('.', '_', $relation)) . '_sort';

        $model = $query->getModel();
        $baseTable = $model->getTable();

        $segments = explode('.', $relation);
        $lastSegment = array_pop($segments);

        $current = $model;
        foreach ($segments as $segment) {
            $current = $current->{$segment}()->getRelated();
        }

        $parentTable = $current->getTable();
        $foreignKey = $current->{$lastSegment}()->getForeignKeyName();

        if (count($segments) > 0) {
            $query->leftJoinRelationship(implode('.', $segments));
        }

        $query->leftJoin(
            "{$parentTable} as {$alias}",
            "{$parentTable}.{$foreignKey}",
            '=',
            "{$alias}.id"
        );

        return $query
            ->orderByRaw("{$alias}.{$column} {$direction} NULLS LAST")
            ->select("{$baseTable}.*");
    }
}
