<?php

declare(strict_types=1);

namespace MSML\Tables;

use ReflectionClass;
use Illuminate\Support\Facades\Route;
use Spatie\LaravelPackageTools\Package;
use MSML\Tables\Commands\MakeTableCommand;
use MSML\Tables\Mappers\MantineReactTableMapper;
use MSML\Tables\Mappers\Contracts\FrontendMapper;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class TablesServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        $package
            ->name('laravel-tables')
            ->hasConfigFile('tables')
            ->hasCommand(MakeTableCommand::class);
    }

    public function packageRegistered(): void
    {
        $this->app->bind(FrontendMapper::class, function () {
            $mapperClass = config('tables.frontend_mapper', MantineReactTableMapper::class);

            return new $mapperClass;
        });

        $this->registerTables();
    }

    public function packageBooted(): void
    {
        Route::bind('table', function ($value, $route) {
            // Support nested paths: users/users-table
            $segments = $route->parameters();

            if (isset($segments['tablePath'])) {
                $value = $segments['tablePath'] . '/' . $value;
            }

            return app("table.{$value}");
        });
    }

    private function registerTables(): void
    {
        $basePath = config('tables.table_class_path');

        if (!$basePath || !is_dir($basePath)) {
            return;
        }

        $this->registerTablesInDirectory($basePath, $basePath);
    }

    private function registerTablesInDirectory(string $directory, string $basePath): void
    {
        $items = scandir($directory);
        $excludedDirs = config('tables.excluded_directories', []);

        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }

            $fullPath = $directory . DIRECTORY_SEPARATOR . $item;

            if (is_dir($fullPath)) {
                if (in_array($item, $excludedDirs, true)) {
                    continue;
                }

                $this->registerTablesInDirectory($fullPath, $basePath);

                continue;
            }

            if (pathinfo($item, PATHINFO_EXTENSION) !== 'php') {
                continue;
            }

            $class = $this->getClassFromFile($fullPath, $basePath);

            if (!class_exists($class)) {
                continue;
            }

            $reflection = new ReflectionClass($class);

            if ($reflection->isSubclassOf(Table::class) && !$reflection->isAbstract()) {
                $this->app->bind("table.{$class::name()}", fn () => new $class);
            }
        }
    }

    private function getClassFromFile(string $filePath, string $basePath): string
    {
        $relativePath = str_replace($basePath, '', $filePath);
        $relativePath = ltrim($relativePath, DIRECTORY_SEPARATOR);
        $relativePath = str_replace('.php', '', $relativePath);

        $relativeNamespace = str_replace(DIRECTORY_SEPARATOR, '\\', $relativePath);

        $baseNamespace = $this->getNamespace($basePath);

        return $baseNamespace . '\\' . $relativeNamespace;
    }

    private function getNamespace(string $path): string
    {
        $relativePath = str_replace(base_path() . DIRECTORY_SEPARATOR, '', $path);
        $parts = explode(DIRECTORY_SEPARATOR, $relativePath);

        return implode('\\', array_map('ucfirst', $parts));
    }
}
