<?php

namespace MSML\OnlinePaymentPlatform;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Http;
use MSML\OnlinePaymentPlatform\Models\Merchant;
use MSML\OnlinePaymentPlatform\Models\Transaction;

class OnlinePaymentPlatform
{
    /**
     * Http client
     *
     * @var Illuminate\Support\Facades\Http;
     */
    private $httpClient;

    /**
     * Create a new instance.
     *
     * @return void
     */
    public function __construct()
    {
        $baseUri = config('online-payment-platform.endpoint');
        $token = config('online-payment-platform.api_key');

        // Rest client
        $this->httpClient = Http::withOptions([
            'base_uri' => $baseUri,
        ])->withToken($token);
    }

    /**
     * Get iDeal issuers.
     *
     * @return JsonResponse
     */
    public function idealIssuers()
    {
        $response = $this->httpClient->get('ideal_issuers');

        return $response->json();
    }

    /**
     * Get merchant based on uid.
     *
     * @param  string  $merchantUid
     * @return JsonResponse
     */
    public function merchant($merchantUid)
    {
        $response = $this->httpClient->get('merchants/' . $merchantUid);

        return $response->json();
    }

    /**
     * Create a new merchant.
     *
     * @param  Merchant  $merchant
     * @return JsonResponse
     */
    public function createMerchant(Merchant $merchant)
    {
        $response = $this->httpClient->post('merchants', $merchant->toArray());

        return $response->json();
    }

    /**
     * Update the given merchant.
     *
     * @param  string  $merchantUid
     * @param  Merchant  $merchant
     * @return JsonResponse
     */
    public function updateMerchant($merchantUid, Merchant $merchant)
    {
        $response = $this->httpClient->post('merchants/' . $merchantUid, $merchant->toArray());

        return $response->json();
    }

    /**
     * Retrieve the merchants payment methods.
     *
     * @param  string  $merchantUid
     * @return JsonResponse
     */
    public function paymentMethods($merchantUid)
    {
        $response = $this->httpClient->get('merchants/' . $merchantUid . '/payment_methods');

        return $response->json();
    }

    /**
     * Create transaction
     *
     * @param  Transaction  $transaction
     * @return JsonResponse
     */
    public function createTransaction(Transaction $transaction)
    {
        $response = $this->httpClient->post('transactions', $transaction->toArray());

        return $response->json();
    }

    /**
     * Update the given transaction.
     *
     * @param  string  $transactionUid
     * @param  string  $escrow_date
     * @return JsonResponse
     */
    public function updateTransaction($transactionUid, $escrow_date)
    {
        $response = $this->httpClient->post('transactions/' . $transactionUid, [
            'escrow_date' => $escrow_date,
        ]);

        return $response->json();
    }

    /**
     * Get transaction based on uid.
     *
     * @param  string  $transactionUid
     * @return JsonResponse
     */
    public function transaction($transactionUid)
    {
        $response = $this->httpClient->get('transactions/' . $transactionUid);

        return $response->json();
    }

    /**
     * Refund a transaction based on uid.
     *
     * @param  string   $transactionUid
     * @param  string   $internalDescription
     * @param  integer  $amount
     * @return JsonResponse
     */
    public function refundTransaction($transactionUid, $internalDescription, $amount)
    {
        $response = $this->httpClient->post('transactions/' . $transactionUid . '/refunds', [
            'internal_reason' => $internalDescription,
            'amount'          => $amount
        ]);

        return $response->json();
    }

    /**
     * Create bank account for merchant
     *
     * @param string $merchantUid
     * @param string $notifyUrl
     * @param string $returnUrl
     * @param boolean $isDefault
     * @param string $reference
     * @return JsonResponse
     */
    public function createBankAccount($merchantUid, $notifyUrl, $returnUrl, $isDefault = true, $reference = '')
    {
        $response = $this->httpClient->post('merchants/' . $merchantUid . '/bank_accounts', [
            'notify_url' => $notifyUrl,
            'return_url' => $returnUrl,
            'is_default' => $isDefault,
            'reference'  => $reference,
        ]);

        return $response->json();
    }

    /**
     * Get merchants bank accounts
     *
     * @param string $merchantUid
     * @return JsonResponse
     */
    public function bankAccounts($merchantUid)
    {
        $response = $this->httpClient->get('merchants/' . $merchantUid . '/bank_accounts');

        return $response->json();
    }
}
